﻿'---------------------------------------------------------------------------------------
' Module    : frmMain.vb
' Author    : A. De Filippo - The Labeljoy Team
' Date      : 2012-09-24
' Purpose   : Sample project of integration between Labeljoy and a third-party application
' License   : Free for private use. Do not reproduce any of this code without written permission.
'             Contact us first at: support@labeljoy.com
' Disclaimer: Use at your own risk.
'---------------------------------------------------------------------------------------

Imports System.IO

Public Class frmMain
    ' Note  : Most of the controls are declared without the "WithEvents" keyword in frmMain.Designer.vb

#Region "Events"

    Private Sub frmMain_FormClosed(ByVal sender As Object, ByVal e As System.Windows.Forms.FormClosedEventArgs) Handles Me.FormClosed
        ' Clean up
        CleanListDataSource(cboCountry)
    End Sub

    Private Sub frmMain_Load(ByVal sender As Object, ByVal e As System.EventArgs) Handles Me.Load

        Dim oDts As DataSet = Nothing
        Try

            ' Ensure needed files are deployed in the proper location
            If DeployResourceFile(My.Resources.Logistic_label_mdb, MdbFileLocation) = False Then
                Throw New Exception("Unable to save MDB file.")
            End If
            If DeployResourceFile(My.Resources.Logistic_label_lpa, LpaFileLocation) = False Then
                ' Error
                Throw New Exception("Unable to save LPA file.")
            End If

            ' Init date to today
            dtpDeliveryDate.Value = Now()

            ' Load country list
            SetListDataSource("SELECT NumericCode, [2DigitCode], Country FROM ISO3166 ORDER BY Country", cboCountry, "Country", "", "NumericCode")

            ' Load last used data if any
            oDts = SqlSelect("SELECT * FROM LogisticLabel")
            If oDts.Tables(0).Rows.Count Then
                With oDts.Tables(0).Rows(0)
                    txtCompanyName.Text = .Item("Company")
                    txtAddress.Text = .Item("Address")
                    txtCity.Text = .Item("City")
                    txtZipCode.Text = .Item("ZipCode")
                    cboCountry.Text = .Item("Country")
                    txtCarrier.Text = .Item("Carrier")
                    dtpDeliveryDate.Value = .Item("DeliveryDate")
                    nudGrossWeight.Value = .Item("GrossWeight")
                    txtOrderNumber.Text = .Item("OrderNumber")
                    TxtSscc.Text = .Item("SSCC")
                End With
            End If

        Catch ex As Exception

            ' Error
            MsgBox("Error: " & ex.Message, MsgBoxStyle.Critical)

            ' Disable print
            btnPrint.Enabled = False

        Finally

            ' Clean up
            DtsClean(oDts)

        End Try

    End Sub

    Private Sub btnPrint_Click(ByVal sender As Object, ByVal e As System.EventArgs) Handles btnPrint.Click

        Try

            ' Check data
            If CheckData() = False Then Exit Sub

            ' Update data in MDB file
            If UpdateData() = False Then Exit Sub

            ' Launch label printing
            LabelPrint(LpaFileLocation)

        Catch ex As Exception
            ' Error
            MsgBox(ex.Message, MsgBoxStyle.Critical)
        End Try

    End Sub

    Private Sub btnQuit_Click(ByVal sender As Object, ByVal e As System.EventArgs) Handles btnQuit.Click
        ' Exit application
        Me.Close()
    End Sub

#End Region

#Region "Resource file deploy"

    ''' <summary>
    ''' Deploy a file from resource to disk
    ''' </summary>
    ''' <param name="Resource">Resource byte array</param>
    ''' <param name="PathAndFileName">Destination path and file name</param>
    Private Function DeployResourceFile(ByVal Resource As Byte(),
                                        ByVal PathAndFileName As String) As Boolean
        ' Check existence
        Try
            If File.Exists(PathAndFileName) = False Then
                System.IO.File.WriteAllBytes(PathAndFileName, Resource)
            End If
        Catch ex As Exception
            ' Error
            Throw New Exception(ex.Message)
        End Try

        ' OK
        Return True

    End Function

#End Region

#Region "Data validation and update"

    ''' <summary>
    ''' Check data consistency.
    ''' </summary>
    ''' <returns>True on data OK.</returns>
    Private Function CheckData() As Boolean

        ' Check company name
        If txtCompanyName.Text.Trim = "" Then
            txtCompanyName.Focus()
            MsgBox("Enter Company name.", MsgBoxStyle.Exclamation)
            Return False
        End If

        ' Check address
        If txtAddress.Text.Trim = "" Then
            txtAddress.Focus()
            MsgBox("Enter address.", MsgBoxStyle.Exclamation)
            Return False
        End If

        ' Check city
        If txtCity.Text.Trim = "" Then
            txtCity.Focus()
            MsgBox("Enter city.", MsgBoxStyle.Exclamation)
            Return False
        End If

        ' Check zip
        If txtZipCode.Text.Trim = "" Then
            txtZipCode.Focus()
            MsgBox("Enter ZIP code.", MsgBoxStyle.Exclamation)
            Return False
        End If

        ' Check country
        If cboCountry.SelectedIndex < 0 Then
            cboCountry.Focus()
            MsgBox("Select country.", MsgBoxStyle.Exclamation)
            Return False
        End If

        ' Check carrier
        If txtCarrier.Text.Trim = "" Then
            txtCarrier.Focus()
            MsgBox("Enter carrier.", MsgBoxStyle.Exclamation)
            Return False
        End If

        ' Check wight
        If nudGrossWeight.Value = 0 Then
            nudGrossWeight.Focus()
            MsgBox("Enter gross weight.", MsgBoxStyle.Exclamation)
            Return False
        End If

        ' Check order number
        If txtOrderNumber.Text.Trim = "" Then
            txtOrderNumber.Focus()
            MsgBox("Enter order number.", MsgBoxStyle.Exclamation)
            Return False
        End If

        ' Check SSCC
        If TxtSscc.Text.Trim = "" Then
            TxtSscc.Focus()
            MsgBox("Enter SSCC.", MsgBoxStyle.Exclamation)
            Return False
        Else

            ' Check for invalid characters
            Dim iChr As Integer = 0
            For iChr = 0 To TxtSscc.Text.Length - 1
                Dim sChr As String = TxtSscc.Text.Substring(iChr, 1)
                Select Case Asc(sChr)
                    Case &H30 To &H39
                        ' OK
                    Case Else
                        ' Not a number
                        TxtSscc.Focus()
                        MsgBox("Invalid character in SSCC field." & vbCrLf & "Only numbers are allowed.", MsgBoxStyle.Exclamation)
                        Return False
                End Select
            Next iChr

            ' Check length
            If TxtSscc.Text.Length <> 17 Then
                TxtSscc.Focus()
                MsgBox("Invalid SSCC, expected length is 17 digits.", MsgBoxStyle.Exclamation)
                Return False
            End If

        End If

        ' Data check OK
        Return True

    End Function

    ''' <summary>
    ''' Update data into the MDB file.
    ''' </summary>
    ''' <returns></returns>
    Private Function UpdateData() As Boolean

        ' Clean up current data
        SqlUpdate("DELETE FROM LogisticLabel")

        ' Get ISO3166 2Digit code from the ComboBox source
        Dim oDtt As DataTable = cboCountry.DataSource
        Dim oRows() As DataRow = oDtt.Select("NumericCode = " & cboCountry.SelectedValue)
        Dim sPrefix As String = oRows(0).Item("2DigitCode")

        ' Insert new data
        SqlUpdate("INSERT INTO LogisticLabel (" & vbLf &
                  "                             Company" & vbLf &
                  "                           , Address" & vbLf &
                  "                           , ZipCode" & vbLf &
                  "                           , City" & vbLf &
                  "                           , Country" & vbLf &
                  "                           , CountryPrefix" & vbLf &
                  "                           , CountryCode" & vbLf &
                  "                           , Carrier" & vbLf &
                  "                           , DeliveryDate" & vbLf &
                  "                           , GrossWeight" & vbLf &
                  "                           , OrderNumber" & vbLf &
                  "                           , SSCC" & vbLf &
                  "                           )" & vbLf &
                  "VALUES (" & vbLf &
                  "          " & DbText(txtCompanyName.Text.Trim) & vbLf &
                  "        , " & DbText(txtAddress.Text.Trim) & vbLf &
                  "        , " & DbText(txtZipCode.Text.Trim) & vbLf &
                  "        , " & DbText(txtCity.Text.Trim) & vbLf &
                  "        , " & DbText(cboCountry.Text.Trim) & vbLf &
                  "        , " & DbText(sPrefix.Trim) & vbLf &
                  "        , " & DbText(Format(cboCountry.SelectedValue, "000")) & vbLf &
                  "        , " & DbText(txtCarrier.Text.Trim) & vbLf &
                  "        , #" & Format(dtpDeliveryDate.Value, "MM/dd/yyyy") & "#" & vbLf &
                  "        , " & nudGrossWeight.Value & vbLf &
                  "        , " & DbText(txtOrderNumber.Text) & vbLf &
                  "        , " & DbText(TxtSscc.Text) & vbLf &
                  "        )")


        ' Data update OK
        Return True

    End Function

#End Region

#Region "Launch printing"

    ' Windows API function that executes a file in the corresponding executable program.
    ' More information: http://msdn.microsoft.com/en-us/library/windows/desktop/bb762153(v=vs.85).aspx
    Private Const SW_SHOW As Integer = 5
    Private Declare Function ShellExecute Lib "shell32.dll" Alias "ShellExecuteA" (ByVal hwnd As Int32, _
                                                                                   ByVal lpOperation As String, _
                                                                                   ByVal lpFile As String, _
                                                                                   ByVal lpParameters As String, _
                                                                                   ByVal lpDirectory As String, _
                                                                                   ByVal nShowCmd As Int32) As Int32


    ' Shell execute return values
    Private Const ERROR_FILE_NOT_FOUND As Int32 = 2&     ' The specified file was not found.
    Private Const ERROR_PATH_NOT_FOUND As Int32 = 3&     ' The specified path was not found.
    Private Const ERROR_BAD_FORMAT As Int32 = 11&        ' The .exe file is invalid (non-Win32 .exe or error in .exe image).
    Private Const SE_ERR_ACCESSDENIED As Int32 = 5       ' The operating system denied access to the specified file.
    Private Const SE_ERR_ASSOCINCOMPLETE As Int32 = 27   ' The file name association is incomplete or invalid.
    Private Const SE_ERR_DDEBUSY As Int32 = 30           ' The DDE transaction could not be completed because other DDE transactions were being processed.
    Private Const SE_ERR_DDEFAIL As Int32 = 29           ' The DDE transaction failed.
    Private Const SE_ERR_DDETIMEOUT As Int32 = 28        ' The DDE transaction could not be completed because the request timed out.
    Private Const SE_ERR_DLLNOTFOUND As Int32 = 32       ' The specified DLL was not found.
    Private Const SE_ERR_FNF As Int32 = 2                ' The specified file was not found.
    Private Const SE_ERR_NOASSOC As Int32 = 31           ' There is no application associated with the given file name extension. This error will also be returned if you attempt to print a file that is not printable.
    Private Const SE_ERR_OOM As Int32 = 8                ' There was not enough memory to complete the operation.
    Private Const SE_ERR_PNF As Int32 = 3                ' The specified path was not found.
    Private Const SE_ERR_SHARE As Int32 = 26             ' A sharing violation occurred.

    ''' <summary>
    ''' Execute a Labeljoy file for printing.
    ''' </summary>
    ''' <param name="LabelJoyFileNameAndPath">
    ''' Full path and file name of the file to print.
    ''' </param>
    ''' <returns>Boolean</returns>
    Private Function LabelPrint(ByVal LabelJoyFileNameAndPath As String) As Boolean

        Try

            ' Check parameter
            If LabelJoyFileNameAndPath = "" Then
                Throw New Exception("Invalid label file.")
            End If

            ' Check that the file exists
            If Dir(LabelJoyFileNameAndPath, vbHidden) = "" Then
                Throw New Exception("Unable to locate file " & LabelJoyFileNameAndPath & ".")
            End If

            ' Print the Labeljoy file
            Select Case ShellExecute(0, "print", LabelJoyFileNameAndPath, "", "", SW_SHOW)
                Case 0
                    Throw New Exception("Out of memory")
                Case ERROR_FILE_NOT_FOUND
                    Throw New Exception("The specified file was not found.")
                Case ERROR_PATH_NOT_FOUND
                    Throw New Exception("The specified path was not found.")
                Case ERROR_BAD_FORMAT
                    Throw New Exception("The .exe file is invalid (non-Win32 .exe or error in .exe image).")
                Case SE_ERR_ACCESSDENIED
                    Throw New Exception("The operating system denied access to the specified file.")
                Case SE_ERR_ASSOCINCOMPLETE
                    Throw New Exception("The file name association is incomplete or invalid.")
                Case SE_ERR_DDEBUSY
                    Throw New Exception("The DDE transaction could not be completed because other DDE transactions were being processed.")
                Case SE_ERR_DDEFAIL
                    Throw New Exception("The DDE transaction failed.")
                Case SE_ERR_DDETIMEOUT
                    Throw New Exception("The DDE transaction could not be completed because the request timed out.")
                Case SE_ERR_DLLNOTFOUND
                    Throw New Exception("The specified DLL was not found.")
                Case SE_ERR_FNF
                    Throw New Exception("The specified file was not found.")
                Case SE_ERR_NOASSOC
                    Throw New Exception("There is no application associated with the given file name extension. This error will also be returned if you attempt to print a file that is not printable.")
                Case SE_ERR_OOM
                    Throw New Exception("There was not enough memory to complete the operation.")
                Case SE_ERR_PNF
                    Throw New Exception("The specified path was not found.")
                Case SE_ERR_SHARE
                    Throw New Exception("A sharing violation occurred.")
                Case Is > SE_ERR_DLLNOTFOUND
                    Return True ' Return True on Success
                Case Else
                    Throw New Exception("Unknown error.")
            End Select

        Catch ex As Exception
            ' Error
            Dim sErrDesc As String = "Error in procedure LabelPrint." & vbCrLf & _
                                     "Error description:" & ex.Message
            MsgBox(sErrDesc, MsgBoxStyle.Critical)
        End Try

    End Function

#End Region

End Class
